# RagStackSetup.ps1
# Install-only GUI for Ollama + optional model pulls (granite3.2:2b, all-minilm)
# - Qdrant removed, no services started by this script
# - Robust Ollama detection (PATH, common folders, WindowsApps shim, registry, service)
# - Winget (machine → user) then direct silent installer fallback
# - Quiet download (BITS; falls back to Invoke-WebRequest; no noisy progress spam)

Add-Type -AssemblyName System.Windows.Forms
Add-Type -AssemblyName System.Drawing
$ErrorActionPreference = 'Stop'
$ProgressPreference = 'SilentlyContinue'  # hide built-in transfer noise

# ---------------- Paths ----------------
$BaseDir = Join-Path $env:LOCALAPPDATA 'rag-stack'
$BinDir  = Join-Path $BaseDir 'bin'
$LogDir  = Join-Path $BaseDir 'logs'
$DataDir = Join-Path $BaseDir 'data'
New-Item -Force -ItemType Directory -Path $BinDir,$LogDir,$DataDir | Out-Null

# ---------------- Config ----------------
$script:OllamaExe = 'ollama'   # will be resolved later

# ---------------- Helpers ----------------
function Log($msg) {
  $ts = (Get-Date).ToString('HH:mm:ss')
  $TextLog.AppendText("[$ts] $msg`r`n")
  $TextLog.ScrollToCaret()
  [System.Windows.Forms.Application]::DoEvents()
}

function Get-RegistryDefaultValue([string]$keyPath) {
  try {
    $k = Get-Item -Path $keyPath -ErrorAction Stop
    return $k.GetValue('')
  } catch { return $null }
}

function Resolve-OllamaPath {
  # 1) PATH
  try {
    $cmd = Get-Command ollama -ErrorAction SilentlyContinue
    if ($cmd -and $cmd.Source) { return $cmd.Source }
  } catch {}

  # 2) Common install folders
  $pf   = $env:ProgramFiles
  $pf86 = ${env:ProgramFiles(x86)}
  $candidates = @()
  if ($env:LOCALAPPDATA) { $candidates += (Join-Path $env:LOCALAPPDATA 'Programs\Ollama\ollama.exe') }
  if ($pf)   { $candidates += (Join-Path $pf   'Ollama\ollama.exe') }
  if ($pf86) { $candidates += (Join-Path $pf86 'Ollama\ollama.exe') }
  # WindowsApps shim (user)
  if ($env:LOCALAPPDATA) { $candidates += (Join-Path $env:LOCALAPPDATA 'Microsoft\WindowsApps\ollama.exe') }

  foreach ($p in $candidates) { if ($p -and (Test-Path $p)) { return $p } }

  # 3) Registry App Paths
  $reg1 = Get-RegistryDefaultValue 'HKLM:\Software\Microsoft\Windows\CurrentVersion\App Paths\ollama.exe'
  if ($reg1 -and (Test-Path $reg1)) { return $reg1 }
  $reg2 = Get-RegistryDefaultValue 'HKCU:\Software\Microsoft\Windows\CurrentVersion\App Paths\ollama.exe'
  if ($reg2 -and (Test-Path $reg2)) { return $reg2 }

  # 4) Service ImagePath
  try {
    $svc = Get-ItemProperty 'HKLM:\SYSTEM\CurrentControlSet\Services\Ollama' -Name ImagePath -ErrorAction SilentlyContinue
    if ($svc.ImagePath) {
      $path = $svc.ImagePath -replace '^\"?(.+?)\"?.*$', '$1'
      if (Test-Path $path) { return $path }
      $maybe = Join-Path (Split-Path -Parent $path) 'ollama.exe'
      if (Test-Path $maybe) { return $maybe }
    }
  } catch {}

  # 5) Quick scan fallback
  $roots = @()
  if ($pf)   { $roots += (Join-Path $pf   'Ollama') }
  if ($pf86) { $roots += (Join-Path $pf86 'Ollama') }
  if ($env:LOCALAPPDATA) { $roots += (Join-Path $env:LOCALAPPDATA 'Programs\Ollama') }
  $usersRoot = Join-Path $env:SystemDrive 'Users'
  if (Test-Path $usersRoot) { $roots += (Join-Path $usersRoot '*\AppData\Local\Programs\Ollama') }

  foreach ($r in $roots) {
    try {
      $found = Get-ChildItem -Path $r -Filter 'ollama.exe' -Recurse -ErrorAction SilentlyContinue |
               Select-Object -First 1 -ExpandProperty FullName
      if ($found) { return $found }
    } catch {}
  }

  return $null
}

function Ensure-Winget() {
  if (-not (Get-Command winget -ErrorAction SilentlyContinue)) {
    throw "winget (App Installer) is required. Install from Microsoft Store, then re-run."
  }
}

function Download-FileQuiet([string]$Url, [string]$DestPath) {
  try {
    Log "Downloading Ollama installer..."
    # Prefer BITS (quiet, resilient)
    Start-BitsTransfer -Source $Url -Destination $DestPath -TransferType Download -ErrorAction Stop
    if (-not (Test-Path $DestPath)) { throw "BITS reported success but file not found." }
    return $true
  } catch {
    Log "BITS download failed: $($_.Exception.Message)"
    try {
      # Fallback to Invoke-WebRequest (quiet due to $ProgressPreference)
      [Net.ServicePointManager]::SecurityProtocol = [Net.SecurityProtocolType]::Tls12
      Invoke-WebRequest -Uri $Url -OutFile $DestPath -UseBasicParsing -ErrorAction Stop
      if (-not (Test-Path $DestPath)) { throw "IWR reported success but file not found." }
      return $true
    } catch {
      Log "Fallback download failed: $($_.Exception.Message)"
      return $false
    }
  }
}

function Install-Ollama-Direct {
  $url = 'https://ollama.com/download/OllamaSetup.exe'
  $dst = Join-Path $env:TEMP 'OllamaSetup.exe'
  if (-not (Download-FileQuiet -Url $url -DestPath $dst)) {
    Log "Direct installer attempt failed: could not download."
    return $null
  }

  # Try common silent switches
  $switches = @('/S','/quiet /norestart','/verysilent /norestart')
  foreach ($sw in $switches) {
    try {
      Log "Running installer silently ($sw)..."
      Start-Process -FilePath $dst -ArgumentList $sw -Wait -PassThru | Out-Null
      Start-Sleep -Seconds 3
      $res = Resolve-OllamaPath
      if ($res) { return $res }
    } catch {
      Log "Silent install attempt failed with switches '$sw': $($_.Exception.Message)"
    }
  }
  return $null
}

function Ensure-Ollama() {
  $resolved = Resolve-OllamaPath
  if (-not $resolved) {
    Log "Ollama not found. Installing via winget (machine scope)..."
    try {
      Start-Process -FilePath "winget" -ArgumentList "install --id=Ollama.Ollama -e --source winget --scope machine --accept-package-agreements --accept-source-agreements --silent" -Wait -PassThru | Out-Null
    } catch {
      Log "Winget machine-scope install threw: $($_.Exception.Message)"
    }
    Start-Sleep -Seconds 3
    $resolved = Resolve-OllamaPath

    if (-not $resolved) {
      Log "Machine-scope install not found. Retrying per-user install..."
      try {
        Start-Process -FilePath "winget" -ArgumentList "install --id=Ollama.Ollama -e --source winget --scope user --accept-package-agreements --accept-source-agreements --silent" -Wait -PassThru | Out-Null
      } catch {
        Log "Winget user-scope install threw: $($_.Exception.Message)"
      }
      Start-Sleep -Seconds 3
      $resolved = Resolve-OllamaPath
    }

    if (-not $resolved) {
      Log "Winget could not produce a detectable executable. Trying direct installer..."
      $resolved = Install-Ollama-Direct
    }

    if ($resolved) {
      $script:OllamaExe = $resolved
      Log "✅ Ollama installed. Using: $resolved"
      $dir = Split-Path -Parent $resolved
      if ($dir -and ($env:PATH -notlike "*$dir*")) { $env:PATH = "$dir;$($env:PATH)" }
    } else {
      throw "Ollama installation reported success but the executable was not found. Please re-run or install manually from https://ollama.com."
    }
  } else {
    $script:OllamaExe = $resolved
    Log "✅ Ollama already installed. Using: $resolved"
  }
}

function Get-InstalledModelNames() {
  try {
    (& $script:OllamaExe list | ForEach-Object { ($_ -split '\s+')[0] }) | Where-Object { $_ } | Sort-Object -Unique
  } catch {
    Log "WARNING: Could not query installed models: $($_.Exception.Message)"
    @()
  }
}

function Ensure-Model([string]$modelName) {
  if (-not $modelName) { return }
  $installed = Get-InstalledModelNames
  if ($installed -contains $modelName) {
    Log "✅ Model already present: $modelName"
    return
  }
  Log "Pulling model (not present): $modelName ..."
  try {
    & $script:OllamaExe pull $modelName
    $installed2 = Get-InstalledModelNames
    if ($installed2 -contains $modelName) {
      Log "✅ Successfully pulled: $modelName"
    } else {
      Log "⚠️ Pull completed but model not detected in list: $modelName"
    }
  } catch {
    Log ("ERROR pulling ${modelName}: " + $_.Exception.Message)
  }
}

# ---------------- GUI ----------------
$form               = New-Object System.Windows.Forms.Form
$form.Text          = "RAG Stack Setup (Install Only: Ollama + optional model pulls)"
$form.Size          = New-Object System.Drawing.Size(820,560)
$form.StartPosition = "CenterScreen"

$lblHeader = New-Object System.Windows.Forms.Label
$lblHeader.Text = "Install prerequisites and optionally pull models (no services started by this script)."
$lblHeader.Location = '20,20'
$lblHeader.AutoSize = $true
$lblHeader.Font = New-Object System.Drawing.Font('Segoe UI', 11, [System.Drawing.FontStyle]::Bold)

$chkGranite = New-Object System.Windows.Forms.CheckBox
$chkGranite.Text = "Ensure model: granite3.2:2b"
$chkGranite.Location = '20,85'
$chkGranite.AutoSize = $true
$chkGranite.Checked = $true

$chkMiniLM = New-Object System.Windows.Forms.CheckBox
$chkMiniLM.Text = "Ensure model: all-minilm (embeddings)"
$chkMiniLM.Location = '20,110'
$chkMiniLM.AutoSize = $true
$chkMiniLM.Checked = $true

$btnInstall = New-Object System.Windows.Forms.Button
$btnInstall.Text = "Check & Install"
$btnInstall.Location = '20,145'
$btnInstall.Width = 160

$Progress = New-Object System.Windows.Forms.ProgressBar
$Progress.Location = '200,147'
$Progress.Width = 580
$Progress.Style = 'Continuous'
$Progress.Minimum = 0
$Progress.Maximum = 4
$Progress.Value = 0

$TextLog = New-Object System.Windows.Forms.TextBox
$TextLog.Multiline = $true
$TextLog.ScrollBars = 'Vertical'
$TextLog.Location = '20,190'
$TextLog.Size = New-Object System.Drawing.Size(760,320)
$TextLog.Font = New-Object System.Drawing.Font('Consolas', 10)

$form.Controls.AddRange(@($lblHeader,$chkGranite,$chkMiniLM,$btnInstall,$Progress,$TextLog))

function Disable-Controls($state) {
  $form.UseWaitCursor = -not $state
  foreach ($c in @($btnInstall,$chkGranite,$chkMiniLM)) { $c.Enabled = $state }
  [System.Windows.Forms.Application]::DoEvents()
}

$btnInstall.Add_Click({
  try {
    Disable-Controls $false
    Log "Starting install checks..."

    $models = @()
    if ($chkGranite.Checked) { $models += 'granite3.2:2b' }
    if ($chkMiniLM.Checked)  { $models += 'all-minilm' }

    $Progress.Value = 0
    $Progress.Maximum = 2 + $models.Count

    Ensure-Winget; $Progress.Value++
    Ensure-Ollama; $Progress.Value++

    foreach ($m in $models) {
      Ensure-Model $m
      $Progress.Value++
    }

    Log "✅ Done. Only installations and model presence checks were performed; no services were started by this script."
  } catch {
    Log ("ERROR: " + $_.Exception.Message)
  } finally {
    Disable-Controls $true
  }
})

[void]$form.ShowDialog()
